#include "ball.hpp"

#include "playstate.hpp"

#include "main.hpp"

Ball::Ball(const BallsyGame *game, const PlayState *playState) : game(game), playState(playState) {
  ballSprite = new Sprite();
  ballSprite->LoadFile("res/gfx/ball_variation.png", e_FilterMode_Linear);
  ballObject = new Object(&playState->GetWorldNode(), ballSprite);
  ballObject->spatial.z = 0.0f;
  ballObject->spatial.scale = Vector2(0.5f, 0.5f);
  playState->RegisterReplayObject(ballObject);

  predictionDuration_ms = 2000;
  float predictionSeconds = predictionDuration_ms / 1000.0f;
  predictionSize = int(ceil(predictionSeconds / (game->GetPhysicsTimeStep_ms() * 0.001f)));

  BallSituation ballSituation; // ball at 0,0; no movement
  for (unsigned int i = 0; i < predictionSize; i++) {
    predictions.push_back(ballSituation);
  }

  //Recalculate();
}

Ball::~Ball() {
  delete ballObject;
  delete ballSprite;
}

void Ball::OnStepPhysicsTime() {
  // this one is done
  predictions.pop_front();

  // insert placeholder
  BallSituation empty;
  predictions.push_back(empty);

  // calculate actual values for placeholder
  Recalculate(predictionSize - 1, predictionSize - 1);

  ballObject->spatial.position = GetPosition();
}

void Ball::Draw(const Viewport &viewport, const Camera &camera) const {
  ballObject->Draw(viewport, camera);
}

const Vector2 &Ball::GetPredictedPosition(unsigned int time_ms) const {
  unsigned int index = int(ceil(std::min(time_ms, GetPredictionDuration_ms() - 1) / (float)game->GetPhysicsTimeStep_ms()));
  if (index >= predictions.size()) {
    //printf("prediction out of bounds (%i, %i, %i)\n", time_ms, index, predictions.size());
    index = predictions.size() - 1;
  }
  //return std::next(predictions.begin(), index)->position;
  return predictions.at(index).position;
}

const Vector2 &Ball::GetPredictedMovement(unsigned int time_ms) const {
  unsigned int index = int(ceil(std::min(time_ms, GetPredictionDuration_ms() - 1) / (float)game->GetPhysicsTimeStep_ms()));
  if (index >= predictions.size()) {
    //printf("prediction out of bounds (%i, %i, %i)\n", time_ms, index, predictions.size());
    index = predictions.size() - 1;
  }
  //return std::next(predictions.begin(), index)->movement;
  return predictions.at(index).movement;
}

void Ball::AddImpulse(const Vector2 &impulse, bool removeCurrentMovement) {
  if (removeCurrentMovement) predictions.front().movement = 0;
  predictions.front().movement += impulse;
  Recalculate(1, predictionSize - 1);
  sig_BallTouched();
  //std::next(predictions.begin(), 1)->movement.Print();
}

BallSituation Ball::IterateOnce(const BallSituation &currentSituation) {

  BallSituation resultSituation;

  //Vector2 resultPosition += this->movement * game->GetPhysicsTimeStep_ms() * 0.001f;
  // if (fabs(position.coords[0]) > 20) this->movement.coords[0] *= -1.0f;
  // if (fabs(position.coords[1]) > 10) this->movement.coords[1] *= -1.0f;

  // friction
  float velocity = currentSituation.movement.GetLength();
  //float friction = 0.2f + 0.1f * velocity; // higher == more
  float friction = 0.4f + 0.05f * velocity; // higher == more

  resultSituation.movement = currentSituation.movement - (currentSituation.movement * friction * game->GetPhysicsTimeStep_ms() * 0.001f).GetNormalizedMax(currentSituation.movement.GetLength()); // normalize so we don't get a pingpong around 0 velo
  resultSituation.position = currentSituation.position + resultSituation.movement * game->GetPhysicsTimeStep_ms() * 0.001f;

  return resultSituation;
}

void Ball::Recalculate(unsigned int startIndex, unsigned int endIndex) {

  int time_ms = 0;

  assert(startIndex > 0);
  assert(endIndex <= predictionSize);

  // grab situation one entry before desired start as base situation
  //std::list<BallSituation>::iterator ballIter = std::next(predictions.begin(), startIndex - 1);
  std::deque<BallSituation>::iterator ballIter = std::next(predictions.begin(), startIndex - 1);

  BallSituation simSituation;
  simSituation.position = ballIter->position;
  simSituation.movement = ballIter->movement;
  ballIter++;

  unsigned int count = 0;
  while (count <= endIndex - startIndex) {
    assert(ballIter != predictions.end());

    simSituation = IterateOnce(simSituation);

    *ballIter = simSituation;
    time_ms += game->GetPhysicsTimeStep_ms();

    count++;
    ballIter++;
  }

}
