#include "playerbody.hpp"

#include "playstate.hpp"
#include "matchteam.hpp"
#include "ball.hpp"
#include "sequences.hpp"
//#include "sequencelibrary.hpp"
//#include "controllers/playercontroller.hpp"

#include "ballsyutils.hpp"

#include "main.hpp"

PlayerBody::PlayerBody(const BallsyGame *game, const PlayState *playState, const MatchTeam *matchTeam, int id) : game(game), playState(playState), matchTeam(matchTeam), id(id) {

  //printf("creating player named %s\n", player->GetLastName().c_str());

  playerSprite = new Sprite();
  playerSprite->LoadFile("res/gfx/player" + patch::to_string(matchTeam->GetID() + 1) + "_SSL.png", e_FilterMode_Linear);
  playerObject = new Object(&playState->GetWorldNode(), playerSprite);
  playerObject->spatial.z = 0.0f;
  playerObject->spatial.scale = Vector2(1.0f, 1.0f);
  playState->RegisterReplayObject(playerObject);

  nameSprite = new Sprite();
  SDL_Color color = { 0, 200, 200 };
  nameSprite->CreateText(game->GetFont(e_FontID_Debug), "player " + patch::to_string(id), color, e_FilterMode_Linear);
  nameObject = new Object(0, nameSprite);
  nameObject->spatial.z = 0.0f;
  nameObject->spatial.scale = Vector2(nameSprite->GetWidth(), nameSprite->GetHeight());
  nameObject->spatial.position = Vector2(0.0f, -1.0f);
  playState->RegisterReplayObject(nameObject);

/*
  // 1-2-1
  if (id == 0) basePosition.Set(-1.0,  0.0);
  if (id == 1) basePosition.Set( 0.3, -0.8);
  if (id == 2) basePosition.Set(-0.3,  0.0);
  if (id == 3) basePosition.Set( 0.3,  0.8);
  if (id == 4) basePosition.Set( 1.0,  0.0);
*/
  // 2-2
  if (id == 0) basePosition.Set(-1.0,  0.0);
  if (id == 1) basePosition.Set(-0.1, -0.3);
  if (id == 2) basePosition.Set(-0.1,  0.3);
  if (id == 3) basePosition.Set( 0.5, -0.8);
  if (id == 4) basePosition.Set( 0.5,  0.8);

  position = GetAbsoluteBasePosition() * Vector2(0.5f, 1.0f); // compress to fit on half
  position.coords[0] += matchTeam->GetSide() * playState->GetStadium()->GetPitchDimensions().coords[0] * 0.25f; // shift offset to own half
  playerObject->spatial.position = position;

  //velocity = 0;
  if (matchTeam->GetSide() == -1) bodyAngle = 0; else bodyAngle = pi;

  currentSequence.sequence = 0;

  playState->GetBall()->sig_BallTouched.connect(boost::bind(&PlayerBody::UpdateSituation, this));
}

PlayerBody::~PlayerBody() {
  delete nameObject;
  delete nameSprite;
  delete playerObject;
  delete playerSprite;
}

void PlayerBody::PreStepPhysicsTime() {
  if (_optimize() && this != matchTeam->GetDesignatedPlayerBody()) {
    int everyX10ms = 1 + 19 * int(floor(NormalizedClamp(this->position.GetDistance(playState->GetBall()->GetPosition()), 0.0f, 20.0f)));
    if (playState->GetGameTime_ms() % everyX10ms * 10 == 0) UpdateSituation();
  } else {
    UpdateSituation();
  }
}

void PlayerBody::OnStepPhysicsTime() {

  unsigned long gameTime_ms = playState->GetGameTime_ms();

  // delete finished sequence?
  if (currentSequence.sequence) {
    if (gameTime_ms > currentSequence.startTime_ms + currentSequence.sequence->GetDuration_ms()) {
      delete currentSequence.sequence;
      currentSequence.sequence = 0;
    }
  }

  if (currentSequence.sequence) {

    //Vector2 movement = currentSequence.sequence->GetMovement(gameTime_ms - currentSequence.startTime_ms);// relative: .GetRotated(currentSequence.startAngle);
    Vector2 newPosition = currentSequence.sequence->GetPosition(gameTime_ms - currentSequence.startTime_ms);// relative: .GetRotated(currentSequence.startAngle);
    this->movement = (newPosition - position) / (game->GetPhysicsTimeStep_ms() * 0.001f);
    if (this->movement.GetLength() > 1.0f) this->bodyAngle = this->movement.GetAngle();
    //this->velocity = movement.GetLength();
    //if (this->velocity > 1.0f) this->angle = movement.GetAngle();
    currentSequence.sequence->ProcessEvents(gameTime_ms - currentSequence.startTime_ms);

  } else {

    Vector2 effectiveDesiredMovement = desiredMovement;

    float maxChangeMPS = 25.0f;
    this->movement = GetPhysicsMovement(GetDirection() * GetVelocity(), effectiveDesiredMovement, maxChangeMPS, game->GetPhysicsTimeStep_ms());
    //this->velocity = movement.GetLength();
    //if (this->velocity > 1.0f) this->angle = movement.GetAngle(); else if (this->desiredMovement.GetLength() > 1.0f) this->angle = this->desiredMovement.GetAngle();
    if (this->movement.GetLength() > 1.0f) this->bodyAngle = this->movement.GetAngle();
    // look at ball this->bodyAngle = (playState->GetBall()->GetPosition() - this->position).GetAngle();

  }


  position += GetDirection() * GetVelocity() * game->GetPhysicsTimeStep_ms() * 0.001f;

  playerObject->spatial.position = position;
  playerObject->spatial.orientation = GetBodyAngle();

  nameObject->spatial.position = WorldToScreenVector(playerObject->GetDerivedSpatial().position + Vector2(0, -1), game->GetMainViewport(), playState->GetCamera());

  //if (matchTeam->GetSelectedPlayer() == this) movement.Print();
}

void PlayerBody::Draw(const Viewport &viewport, const Camera &camera) const {
  playerObject->Draw(viewport, camera);
  nameObject->Draw();
}

void PlayerBody::UpdateSituation() {
  //printf("pling! ");
  timeNeededToGetToBall_ms = GetTimeNeededForTravel_ms(true);
}

Vector2 PlayerBody::GetBasePosition() const {
  return basePosition;
}

Vector2 PlayerBody::GetAbsoluteBasePosition() const {
  Vector2 pitchSize = playState->GetStadium()->GetPitchDimensions();
  return basePosition * Vector2(-matchTeam->GetSide(), 1.0f) * pitchSize * 0.5f * 0.9f; // margin
}

//bool PlayerBody::RequestSequence(const Sequence *sequence) {
bool PlayerBody::RequestDribbleSequence(const Vector2 &targetMovement) {
  if (!currentSequence.sequence && DribbleSequence::IsBallTouchable(this, playState->GetBall())) {
    //Vector2 expectedBallPositionTemp = position + Vector2(1, 0).GetRotated(angle) * velocity;//playState->GetBall()->GetPredictedPosition(200);
    //Sequence *createdSequence = playState->GetSequenceLibrary()->CreateSequence(e_SequenceType_Dribble, GetPosition(), GetDirection(), GetVelocity(), targetDirection, targetVelocity);//sequenceLibrary->FindDribbleSequence(playState->GetBall(), playerBody->GetPosition(), playerBody->GetMovement(), inputDirection * inputVelocity);
    DribbleSequence *createdSequence = new DribbleSequence(game, this, playState->GetBall(), targetMovement);
    //if (createdSequence->TestIfLegal()) {// && (playState->GetBall()->GetPredictedPosition(200)).GetDistance(position + Vector2(1, 0).GetRotated(angle) * velocity) < 0.75f) {
    currentSequence.startTime_ms = playState->GetGameTime_ms();
    // relative: currentSequence.startAngle = GetAngle();
    currentSequence.sequence = createdSequence;
    return true;
  } else {
    return false;
  }
}

bool PlayerBody::RequestPassSequence(const Vector2 &targetDirection, float power) {
    if (!currentSequence.sequence && PassSequence::IsBallTouchable(this, playState->GetBall())) {
    //Vector2 expectedBallPositionTemp = position + Vector2(1, 0).GetRotated(angle) * velocity;//playState->GetBall()->GetPredictedPosition(200);
    PassSequence *createdSequence = new PassSequence(game, this, playState->GetBall(), targetDirection, power);
    //Sequence *createdSequence = playState->GetSequenceLibrary()->CreateSequence(e_SequenceType_Pass, GetPosition(), GetDirection(), GetVelocity(), targetDirection, power);//sequenceLibrary->FindDribbleSequence(playState->GetBall(), playerBody->GetPosition(), playerBody->GetMovement(), inputDirection * inputVelocity);
    //if (createdSequence->TestIfLegal()) {// && (playState->GetBall()->GetPredictedPosition(200)).GetDistance(position + Vector2(1, 0).GetRotated(angle) * velocity) < 0.75f) {
    currentSequence.startTime_ms = playState->GetGameTime_ms();
    // relative: currentSequence.startAngle = GetAngle();
    currentSequence.sequence = createdSequence;
    return true;
  } else {
    return false;
  }
}

bool PlayerBody::HasActiveSequence() const {
  if (currentSequence.sequence) return true; else return false;
}

unsigned int PlayerBody::GetTimeNeededForTravel_ms(bool targetBall, const Vector2 &targetPosition) const {

  unsigned int maxTime_ms = playState->GetBall()->GetPredictionDuration_ms();

  Vector2 myPosition = GetPosition();
  float radius = playState->GetBallRadius();
  float radiusBias = 0.0f; // 0.0 == dependent on current movement; 1.0 == full desired direction (use radius)
  float maxVelocity = playState->GetFullVelocity();
  unsigned int changeTime_ms = 400; // time it takes (roughly) to change movement, worst case

  // sane default for long distances: at least maxtime, and then some for the distance; not to get a realistic time, but to make the distiction with others
  unsigned int resultTime_ms = maxTime_ms + GetPosition().GetDistance(playState->GetBall()->GetPredictedPosition(maxTime_ms)) / maxVelocity * 1000.0f;

  float initialDistance = targetBall ? playState->GetBall()->GetPredictedPosition(0).GetDistance(myPosition) : targetPosition.GetDistance(myPosition);
  unsigned int timeStep_ms = 10;
  if (_optimize() && this != matchTeam->GetDesignatedPlayerBody()) timeStep_ms = 10 + (unsigned int)(10 + Clamp((initialDistance - 2.0f) * 10.0f, 0, 200));
  for (unsigned int time_ms = 0; time_ms < maxTime_ms; time_ms += timeStep_ms) {
    Vector2 adaptedTargetPosition = targetPosition;
    if (targetBall) {
      const Vector2 &ballPosition = playState->GetBall()->GetPredictedPosition(time_ms);
      adaptedTargetPosition = ballPosition;
    }

    radiusBias = Clamp(time_ms / (float)changeTime_ms, 0.0f, 1.0f);
    //radiusBias = pow(radiusBias, 2.0f);
    myPosition += GetMovement() * timeStep_ms * 0.001f * (1.0f - radiusBias);
    radius += maxVelocity * timeStep_ms * 0.001f * radiusBias;

    const float &distance = myPosition.GetDistance(adaptedTargetPosition);
    if (distance <= radius) {
      resultTime_ms = time_ms;
      break;
    }

    //timeStep_ms = std::max(timeStep_ms, (unsigned int)(10 + Clamp((distance - 2.0f) * 10.0f, 0, 200))); // low res timestep for greater distances

  }

  return resultTime_ms;
}
