#include "playstate.hpp"

#include "main.hpp"

#include "ball.hpp"
#include "matchteam.hpp"
//#include "playerbody.hpp"
//#include "controllers/humanplayercontroller.hpp"
//#include "sequencelibrary.hpp"

PlayState::PlayState(BallsyGame *game) : State(game) {

  if (game->GetPhysicsTimeStep_ms() % 5 != 0) printf("PHYSICS TIMESTEP MUST BE A MULTIPLE OF 5\n");

  const Viewport &mainViewport = game->GetMainViewport();

  camera.spatial.position = Vector2(0, 0);
  camera.spatial.z = 50.0f;
  camera.spatial.orientation = 0.0f;
  camera.fov = 0.25f * pi;

  hudSprite = new Sprite();
  hudSprite->LoadFile("res/gfx/hud.png", e_FilterMode_Linear);
  hudObject = new Object(0, hudSprite);
  hudObject->spatial.position = Vector2(mainViewport.w / 2, mainViewport.h / 2);
  hudObject->spatial.scale = Vector2(mainViewport.w, mainViewport.h);

  stadium = new Stadium(game, this);
  // std::vector<Object*> stadiumObjects = stadium->GetReplayObjects();
  // replayObjects.insert(replayObjects.end(), stadiumObjects.begin(), stadiumObjects.end());

  ball = new Ball(game, this);

  // dit moet eigenlijk in een resource manager / object store, maar voorlopig ff hier
  clubs[0] = new Club(0); // db id
  clubs[1] = new Club(1);

  matchTeams[0] = new MatchTeam(game, this, clubs[0], 0);
  matchTeams[1] = new MatchTeam(game, this, clubs[1], 1);

  //sequenceLibrary = new SequenceLibrary(game, this);

  bazkie = new HumanGamer(game); // test gamer ^_^ he's so hawt right now
  matchTeams[0]->RegisterHumanGamer(bazkie);

  gameTime_ms = 0;
  runState = e_RunState_Running;
}

PlayState::~PlayState() {

  delete hudObject;
  delete hudSprite;

  delete stadium;

  delete ball;

  //delete sequenceLibrary;

  delete bazkie;

  delete matchTeams[0];
  delete matchTeams[1];
  delete clubs[0];
  delete clubs[1];

}

void PlayState::OnActivate() {
}

void PlayState::OnDeactivate() {
}

void PlayState::OnStepPhysicsTime() {

  if (runState == e_RunState_Running) {
    ball->OnStepPhysicsTime();

    matchTeams[0]->OnStepPhysicsTime();
    matchTeams[1]->OnStepPhysicsTime();
  }

  // player view
  //Vector2 pos = matchTeams[0]->GetSelectedPlayer()->GetPosition() * 0.6f;
  // ball + player view
  //x Vector2 pos = matchTeams[0]->GetSelectedPlayer()->GetPosition() * 0.4f + ball->GetPosition() * 0.2f;
  //Vector2 pos = matchTeams[0]->GetPlayerBody(4)->GetPosition() * 0.4f + ball->GetPosition() * 0.2f;
  //Vector2 pos = matchTeams[0]->GetSelectedPlayer()->GetPosition() * 0.1f + ball->GetPosition() * 0.5f;
  // ball view
  Vector2 pos = ball->GetPosition() * 0.6f;

  camPosHistory.push_back(pos);
  float history_seconds = 0.8f;
  if (camPosHistory.size() > (1000.0f / game->GetPhysicsTimeStep_ms()) * history_seconds) camPosHistory.pop_front();

  std::deque<Vector2>::iterator camPosIter = camPosHistory.begin();
  Vector2 average;
  while (camPosIter != camPosHistory.end()) {
    average += *camPosIter;
    camPosIter++;
  }
  average /= camPosHistory.size();

  camera.spatial.position = average;

  if (runState == e_RunState_Running) {
    gameTime_ms += game->GetPhysicsTimeStep_ms();
    SnapshotReplayObjects();
    CleanOutdatedReplayLines();
    CleanOutdatedReplayTexts();
  }
}

void PlayState::OnStepGraphicsTime() {

  stadium->Draw(game->GetMainViewport(), GetCamera());

  ball->Draw(game->GetMainViewport(), GetCamera());

  //matchTeams[0]->OnStepGraphicsTime();
  //matchTeams[1]->OnStepGraphicsTime();
  matchTeams[0]->Draw(game->GetMainViewport(), GetCamera());
  matchTeams[1]->Draw(game->GetMainViewport(), GetCamera());

  hudObject->Draw();

}

void PlayState::DrawDebugLine(const Vector2 &p1, const Vector2 &p2, float r, float g, float b, float alpha) const {
  Vector2 p1screen = WorldToScreenVector(p1, game->GetMainViewport(), GetCamera());
  Vector2 p2screen = WorldToScreenVector(p2, game->GetMainViewport(), GetCamera());
  SDL_SetRenderDrawColor(GetSDLRenderer(), r * 255.0f, g * 255.0f, b * 255.0f, alpha * 255.0f);
  SDL_RenderDrawLine(GetSDLRenderer(), p1screen.coords[0], p1screen.coords[1], p2screen.coords[0], p2screen.coords[1]);

  // struct ReplayLine {
  //   int timestamp_ms;
  //   float r, g, b, alpha;
  //   LineCoords coords;
  // };

  if (runState == e_RunState_Running) {
    ReplayLine replayLine;
    replayLine.timestamp_ms = GetGameTime_ms();
    replayLine.p1 = p1;
    replayLine.p2 = p2;
    replayLine.r = r;
    replayLine.g = g;
    replayLine.b = b;
    replayLine.alpha = alpha;
    replayLines.push_back(replayLine);
  }

}

void PlayState::DrawDebugText(const std::string &text, const Vector2 &position, int debugDuration_ms) const {

  Sprite *textSprite = new Sprite();
  SDL_Color color = { 200, 50, 50 };
  textSprite->CreateText(game->GetFont(e_FontID_Debug), text, color, e_FilterMode_Linear);
  Spatial spatial;
  spatial.position = WorldToScreenVector(position, game->GetMainViewport(), GetCamera());
  //spatial.position = position;
  spatial.scale.coords[0] = textSprite->GetWidth();
  spatial.scale.coords[1] = textSprite->GetHeight();
  textSprite->Draw(spatial);
  delete textSprite;

  if (runState == e_RunState_Running) {
    ReplayText replayText;
    replayText.text = text;
    replayText.position = position;
    replayText.timestamp_ms = GetGameTime_ms();
    replayText.duration_ms = (debugDuration_ms == -1) ? game->GetPhysicsTimeStep_ms() : (unsigned int)debugDuration_ms;
    replayTexts.push_back(replayText);
  }
}

void PlayState::SnapshotReplayObjects() {

  float replay_seconds = GetMaxReplayDuration_ms() / (float)1000.0f;

  for (unsigned int i = 0; i < replayObjects.size(); i++) {

    SpatialSnapshot snapshot;
    snapshot.timestamp_ms = GetGameTime_ms();
    snapshot.spatial = replayObjects.at(i).object->GetDerivedSpatial();

    replayObjects.at(i).spatialSnapshots.push_back(snapshot);

    if (replayObjects.at(i).spatialSnapshots.size() > (1000.0f / (float)game->GetPhysicsTimeStep_ms()) * replay_seconds) replayObjects.at(i).spatialSnapshots.pop_front();

  }

}

void PlayState::CleanOutdatedReplayLines() {
  // while (!replayLines.empty() && (signed long)replayLines.front().timestamp_ms < (signed long)(GetGameTime_ms() - GetMaxReplayDuration_ms())) {
  while (!replayLines.empty() && replayLines.front().timestamp_ms < GetGameTime_ms() - std::min((unsigned long)GetMaxReplayDuration_ms(), GetGameTime_ms()) ) { // std::min to stop overflow
    replayLines.pop_front();
  }
}

void PlayState::CleanOutdatedReplayTexts() {
  while (!replayTexts.empty() && replayTexts.front().timestamp_ms < GetGameTime_ms() - std::min((unsigned long)GetMaxReplayDuration_ms(), GetGameTime_ms()) ) { // std::min to stop overflow
    replayTexts.pop_front();
  }
}

/*
SequenceLibrary *PlayState::GetSequenceLibrary() const {
  return sequenceLibrary;
}
*/


// --- HUMANGAMER -------------------------------------------------------------------------------------------------------------------

HumanGamer::HumanGamer(const BallsyGame *game) : game(game) {
}

HumanGamer::~HumanGamer() {
}

const KeyInput &HumanGamer::GetKeyInput() const {
  return game->GetKeyInput();
}


// --- CLUB -------------------------------------------------------------------------------------------------------------------------

Club::Club(int databaseID) : databaseID(databaseID) {
  if (databaseID == 0) clubName = "Properly Decent FC";
  if (databaseID == 1) clubName = "VV Nooblettes";
}

Club::~Club() {
}

/*
MatchTeam *Club::CreateMatchTeam(const Game *game, const State *state) {
  MatchTeam *matchTeam = new MatchTeam(game, state);

  return matchTeam;
}
*/


// --- PLAYER -----------------------------------------------------------------------------------------------------------------------

Player::Player(int databaseID) : playerBody(0) {
  playerData = new PlayerData(databaseID);
}

Player::~Player() {
  delete playerData;
}

PlayerData *Player::GetPlayerData() const {
  return playerData;
}

PlayerBody *Player::GetPlayerBody() const {
  return playerBody;
}

void Player::AttachPlayerBody(PlayerBody *playerBody) {
  assert(!this->playerBody);
  this->playerBody = playerBody;
}

void Player::DetachPlayerBody() {
  assert(this->playerBody);
  this->playerBody = 0;
}


// --- PLAYERDATA -------------------------------------------------------------------------------------------------------------------

PlayerData::PlayerData(int databaseID) : databaseID(databaseID) {
  if (databaseID == 0) lastName = "grave";
  if (databaseID == 1) lastName = "broertjes";
  if (databaseID == 2) lastName = "grevink";
  if (databaseID == 3) lastName = "talle";
  if (databaseID == 4) lastName = "konings";
  if (databaseID == 5) lastName = "noobie";
  if (databaseID == 6) lastName = "nooboz";
  if (databaseID == 7) lastName = "noblitz";
  if (databaseID == 8) lastName = "nibloob";
  if (databaseID == 9) lastName = "nablab";

  if (databaseID < 5) databaseClubID = 0; else databaseClubID = 1;
}

PlayerData::~PlayerData() {
}


// --- STADIUM ----------------------------------------------------------------------------------------------------------------------

Stadium::Stadium(const BallsyGame *game, const PlayState *playState) : game(game), playState(playState) {

  pitchDimensions.Set(40, 20);

  pitchSprite = new Sprite();
  pitchSprite->LoadFile("res/gfx/pitch_grass.png", e_FilterMode_Linear);
  pitchObject = new Object(&playState->GetWorldNode(), pitchSprite);
  pitchObject->spatial.z = -2.0f;
  pitchObject->spatial.position = Vector2(0, 0);
  pitchObject->spatial.scale = pitchDimensions;
  if (!_verbose()) playState->RegisterReplayObject(pitchObject); // in verbose mode, leave pitch transparent, for debug lines' visibility and such

  pitchGlassSprite = new Sprite();
  pitchGlassSprite->LoadFile("res/gfx/glassplate.png", e_FilterMode_Linear);
  pitchGlassObject = new Object(&playState->GetWorldNode(), pitchGlassSprite);
  pitchGlassObject->spatial.z = 0.0f;
  pitchGlassObject->spatial.position = Vector2(0, 0);
  pitchGlassObject->spatial.scale = pitchDimensions;
  playState->RegisterReplayObject(pitchGlassObject);

}

Stadium::~Stadium() {
  delete pitchObject;
  delete pitchSprite;
  delete pitchGlassObject;
  delete pitchGlassSprite;
}

/*
std::vector<Object*> Stadium::GetReplayObjects() {
  std::vector<Object*> objects;
  objects.push_back(pitchObject);
  objects.push_back(pitchGlassObject);
  return objects;
}
*/

/*
void Stadium::OnStepGraphicsTime() {
  pitchObject->Draw(game->GetMainViewport(), playState->GetCamera());
  pitchGlassObject->Draw(game->GetMainViewport(), playState->GetCamera());
}
*/

void Stadium::Draw(const Viewport &viewport, const Camera &camera) const {
  if (!_verbose()) pitchObject->Draw(viewport, camera); // in verbose mode, leave pitch transparent, for debug lines' visibility and such
  pitchGlassObject->Draw(viewport, camera);
}
