#include "replaystate.hpp"

#include "main.hpp"

ReplayState::ReplayState(BallsyGame *game, const std::vector<ReplayObject> &replayObjects, const std::deque<ReplayLine> &replayLines, const std::deque<ReplayText> &replayTexts) : State(game), replayObjects(replayObjects), replayLines(replayLines), replayTexts(replayTexts) {

  //const Viewport &mainViewport = game->GetMainViewport();

  camera.spatial.position = Vector2(0, 0);
  camera.spatial.z = 60.0f;
  camera.spatial.orientation = 0.0f;
  camera.fov = 0.25f * pi;

  replayTime_ms = 0;

}

ReplayState::~ReplayState() {
}

void ReplayState::OnActivate() {
  replayTime_ms = FindEarliestReplayObjectTimestamp_ms();
}

void ReplayState::OnDeactivate() {
  camPosHistory.clear();
}

void ReplayState::OnStepPhysicsTime() {

  Vector2 pos;

  camPosHistory.push_back(pos);
  float history_seconds = 0.5f;
  if (camPosHistory.size() > 1.0f / (game->GetPhysicsTimeStep_ms() * 0.001f) * history_seconds) camPosHistory.pop_front();

  std::deque<Vector2>::iterator camPosIter = camPosHistory.begin();
  Vector2 average;
  while (camPosIter != camPosHistory.end()) {
    average += *camPosIter;
    camPosIter++;
  }
  average /= camPosHistory.size();

  camera.spatial.position = average;


  float speedMultiplier = 1.0f;
  if (static_cast<BallsyGame*>(game)->GetKeyInput().actionkeys[8]) speedMultiplier = 0.5f;
  if (static_cast<BallsyGame*>(game)->GetKeyInput().actionkeys[0]) speedMultiplier = 1.0f / (float)game->GetPhysicsTimeStep_ms(); // slowest possible
  replayTime_ms += round(game->GetPhysicsTimeStep_ms() * speedMultiplier);

}

void ReplayState::OnStepGraphicsTime() {

  bool found = false;

  for (unsigned int i = 0; i < replayObjects.size(); i++) {

    if (replayObjects.at(i).spatialSnapshots.size() > 0) {
      std::deque<SpatialSnapshot>::const_iterator snapshotIter = replayObjects.at(i).spatialSnapshots.begin();
      while (snapshotIter != replayObjects.at(i).spatialSnapshots.end()) {
        if (snapshotIter->timestamp_ms >= replayTime_ms) {

          //const Spatial &absoluteSpatial = snapshotIter->spatial;
          const Sprite *sprite = replayObjects.at(i).object->GetSprite();
          sprite->Draw(snapshotIter->spatial, game->GetMainViewport(), GetCamera());
          found = true;

          break;
        }
        snapshotIter++;
      }
    }

  }

  // debug lines
  for (unsigned int i = 0; i < replayLines.size(); i++) {
    if (replayLines.at(i).timestamp_ms >= replayTime_ms && replayLines.at(i).timestamp_ms < replayTime_ms + game->GetPhysicsTimeStep_ms()) {
      Vector2 p1screen = WorldToScreenVector(replayLines.at(i).p1, game->GetMainViewport(), GetCamera());
      Vector2 p2screen = WorldToScreenVector(replayLines.at(i).p2, game->GetMainViewport(), GetCamera());
      SDL_SetRenderDrawColor(GetSDLRenderer(), replayLines.at(i).r * 255.0f, replayLines.at(i).g * 255.0f, replayLines.at(i).b * 255.0f, replayLines.at(i).alpha * 255.0f);
      SDL_RenderDrawLine(GetSDLRenderer(), p1screen.coords[0], p1screen.coords[1], p2screen.coords[0], p2screen.coords[1]);
    }
  }

  // debug texts
  for (unsigned int i = 0; i < replayTexts.size(); i++) {
    //printf("%i, %i, %i, %i AND %i == %i\n", replayTexts.at(i).timestamp_ms, replayTime_ms, replayTexts.at(i).duration_ms, replayTexts.at(i).timestamp_ms >= replayTime_ms, replayTexts.at(i).timestamp_ms < (unsigned long)(replayTime_ms + replayTexts.at(i).duration_ms), replayTexts.at(i).timestamp_ms >= replayTime_ms && replayTexts.at(i).timestamp_ms < (unsigned long)(replayTime_ms + replayTexts.at(i).duration_ms));
    if (replayTexts.at(i).timestamp_ms >= replayTime_ms && replayTexts.at(i).timestamp_ms < (unsigned long)(replayTime_ms + replayTexts.at(i).duration_ms)) {
      //Vector2 p1screen = WorldToScreenVector(replayLines.at(i).p1, game->GetMainViewport(), GetCamera());
      //Vector2 p2screen = WorldToScreenVector(replayLines.at(i).p2, game->GetMainViewport(), GetCamera());
      Sprite *textSprite = new Sprite();
      SDL_Color color = { 200, 50, 50 };
      textSprite->CreateText(game->GetFont(e_FontID_Debug), replayTexts.at(i).text, color, e_FilterMode_Linear);
      Spatial spatial;
      //spatial.position = replayTexts.at(i).position;
      spatial.position = WorldToScreenVector(replayTexts.at(i).position, game->GetMainViewport(), GetCamera());
      spatial.scale.coords[0] = textSprite->GetWidth();
      spatial.scale.coords[1] = textSprite->GetHeight();
      textSprite->Draw(spatial);
      delete textSprite;
    }
  }

  if (!found) {
    // reset at the end of the replay
    replayTime_ms = FindEarliestReplayObjectTimestamp_ms();
  }

  //printf("desired snapshot time: %lu\n", replayTime_ms);
}

unsigned long ReplayState::FindEarliestReplayObjectTimestamp_ms() {
  if (replayObjects.size() > 0) {
    if (replayObjects.at(0).spatialSnapshots.size() > 0) {
      return replayObjects.at(0).spatialSnapshots.front().timestamp_ms;
    }
  }
  return 0;
}

/*
void ReplayState::SetReplayObjects(const std::vector<ReplayObject> &replayObjects) {
  this->replayObjects = replayObjects;
  replayTime_ms = FindEarliestReplayObjectTimestamp_ms();
}
*/
