#include "sequences.hpp"

#include "main.hpp"

#include "ball.hpp"
#include "playerbody.hpp"

#include "ballsyutils.hpp"


// EVENT: BALL IMPULSE

BallImpulseSequenceEvent::BallImpulseSequenceEvent(Ball *ball, const Vector2 &expectedBallPosition, float maxBallPositionDeviationRadius, const Vector2 &impulse) : SequenceEvent(e_SequenceEventType_BallImpulse), ball(ball), expectedBallPosition(expectedBallPosition), maxBallPositionDeviationRadius(maxBallPositionDeviationRadius), impulse(impulse) {
}

BallImpulseSequenceEvent::~BallImpulseSequenceEvent() {
}

bool BallImpulseSequenceEvent::TestIfLegal(unsigned int futureTime_ms) const {
  //Vector2 predictedBallPosition = ball->GetPosition() + ball->GetMovement() * futureTime_ms * 0.001f;
  Vector2 predictedBallPosition = ball->GetPredictedPosition(futureTime_ms);
  if (expectedBallPosition.GetDistance(predictedBallPosition) > maxBallPositionDeviationRadius) return false; else return true;
}

void BallImpulseSequenceEvent::Execute() {
  if (TestIfLegal(0)) {
    ball->AddImpulse(impulse);
  }
}

/*
int BallImpulseSequenceEvent::GetTriggerTime_ms() const {
  return 100;
}
*/

// /events


// SEQUENCES

Vector2 Sequence::GetPosition(unsigned int time_ms) const {
  //unsigned int index = int(ceil(std::min(time_ms, GetDuration_ms() - game->GetPhysicsTimeStep_ms()) / (float)game->GetPhysicsTimeStep_ms())); // todonow ! +1 ? -1 ? huuu
  unsigned int index = int(ceil(std::min(time_ms, GetDuration_ms()) / (float)game->GetPhysicsTimeStep_ms())); // todonow ! +1 ? -1 ? huuu
  //printf("%i of %i, %i of %i, %f\n", index, positions.size(), time_ms, GetDuration_ms(), (float)game->GetPhysicsTimeStep_ms());
  assert(index < positions.size());
  return positions.at(index);
}

Vector2 Sequence::GetMovement(unsigned int time_ms) const { // todo: test
  printf("time: %i,\n", time_ms);
  return (this->GetPosition(time_ms + game->GetPhysicsTimeStep_ms()) - this->GetPosition(time_ms)) * (1000 / (float)game->GetPhysicsTimeStep_ms());
}

Vector2 Sequence::GetOutgoingMovement() const {
  return ( positions.at(positions.size() - 1) - positions.at(positions.size() - 2) ) * (1000 / (float)game->GetPhysicsTimeStep_ms());
}

void Sequence::ProcessEvents(unsigned int time_ms) const {
  for (unsigned int i = 0; i < events.size(); i++) {
    if (!events.at(i).done && time_ms >= events.at(i).triggerTime_ms && time_ms < events.at(i).triggerTime_ms + game->GetPhysicsTimeStep_ms()) {
      events.at(i).sequenceEvent->Execute();
      //x events.at(i).done = true;
    }
  }
}





void GeneratePhysicsMovement(std::vector<Vector2> &positions, const Game *game, unsigned int duration, const Vector2 &startPosition, const Vector2 &incomingMovement, const Vector2 &outgoingMovement, float exp, float curve) {

  positions.clear();
  Vector2 position = startPosition;

  for (unsigned int time_ms = 0; time_ms <= duration; time_ms += game->GetPhysicsTimeStep_ms()) {

    float frameBias = time_ms / ((float)duration - game->GetPhysicsTimeStep_ms()); // minus one timestep because we want the last bias to be 1.0, not 0.99something

    // curve alterations
    if (exp < 0.999f || exp > 1.001f) frameBias = pow(frameBias, exp);
    if (curve > 0.001f) {
      frameBias = Curve(frameBias, curve);
      frameBias = Curve(frameBias, curve);
    }
    Vector2 movement = incomingMovement * (1.0f - frameBias) + outgoingMovement * frameBias;

    /*
    // brake during cornering
    frameBias = time_ms / ((float)GetDuration_ms() - game->GetPhysicsTimeStep_ms()); // minus one timestep because we want the last bias to be 1.0, not 0.99something
    float movementChangeAmount = NormalizedClamp(incomingMovement.GetDistance(outgoingMovement), 0.0f, 10.0f); // todo: can haz full velo?
    //float brakeInMiddleBias = 0.25f + movementChangeAmount * 0.5f;
    float brakeInMiddleBias = 0.0f + movementChangeAmount * 0.5f;
    movement *= (cos(frameBias * 2.0f * pi) * 0.5f + 0.5f) * brakeInMiddleBias + (1.0f - brakeInMiddleBias);
    */

    position += movement * game->GetPhysicsTimeStep_ms() * 0.001f;
    positions.push_back(position);
  }

}





/*
// MOVEMENT

MovementSequence::MovementSequence(const BallsyGame *game, const Vector2 &test) : Sequence(game, e_SequenceType_Movement), test(test) {
}

MovementSequence::~MovementSequence() {
}

bool MovementSequence::TestIfLegal() const {
  return true;
}

Vector2 MovementSequence::GetMovement(unsigned int time_ms) const {
  //return Vector2(sin(time_ms / (float)GetDuration_ms() * 2 * pi), cos(time_ms / (float)GetDuration_ms() * 2 * pi));
  //return Vector2(0);
  return test;
}

unsigned int MovementSequence::GetDuration_ms() const {
  return 250;
}
*/


// DRIBBLE

float DribbleSequence::maxBallPositionTouchRadius = 0.25f;
float DribbleSequence::touchTime_ms = 100;
float DribbleSequence::maxBallPositionDeviationRadius = 0.25f;
DribbleSequence::DribbleSequence(const BallsyGame *game, const PlayerBody *playerBody, Ball *ball, const Vector2 &desiredMovement) : Sequence(game, e_SequenceType_Dribble, playerBody->GetMovement()) {

  Vector2 maximizedDesiredMovement = desiredMovement.EnforceMaximumDeviation(incomingMovement, 10.0f);//7.5f);

  float diff = NormalizedClamp(incomingMovement.GetDistance(maximizedDesiredMovement), 0.0f, 10.0f);

  float exp = 0.3f + diff * 0.7f;
  float curve = 0.8f;
  GeneratePhysicsMovement(positions, game, GetDuration_ms(), playerBody->GetPosition(), playerBody->GetMovement(), maximizedDesiredMovement, exp, curve);

  Vector2 expectedBallPosition = ball->GetPredictedPosition(DribbleSequence::touchTime_ms);
/*
  //Vector2 preferredBallPosition = DribbleSequence::GetPreferredBallPosition(playerBody);

  // only correct for incoming movement if it's a lot different from desired outgoing
  float ignoreIncomingOutgoingDifferenceAmount = 5.0f;//7.5f;
  float power = 0.3f;
  Vector2 incomingMovementCorrection;
  Vector2 movementChange = (GetOutgoingMovement() - GetIncomingMovement());
  if (movementChange.GetLength() > ignoreIncomingOutgoingDifferenceAmount) {
    float currentMovementCorrectionBias = NormalizedClamp(movementChange.GetLength() - ignoreIncomingOutgoingDifferenceAmount, 0.0f, 7.5f);
    power *= 1.0f - currentMovementCorrectionBias;
    incomingMovementCorrection = GetIncomingMovement() * currentMovementCorrectionBias * 0.9f;
  }
  Vector2 ballImpulse = incomingMovementCorrection + outgoingMovement * (1.0f + power);
*/

  float nextAnimLength = GetOutgoingVelocity() * DribbleSequence::touchTime_ms * 0.001f;
  //float ffoLength = GetOutgoingVelocity() * DribbleSequence::touchTime_ms * 0.001f;
  float ffoLength = DribbleSequence::maxBallPositionTouchRadius;
  Vector2 desiredBallPosition = GetPosition(GetDuration_ms()) + GetOutgoingMovement().GetNormalized(0) * (nextAnimLength + ffoLength);
  unsigned int ffoTime_ms = DribbleSequence::touchTime_ms;

  Vector2 sequenceTranslation = (desiredBallPosition - expectedBallPosition);// * sequenceTime_ms * 0.001f;
  unsigned int sequenceTime_ms = GetDuration_ms() - DribbleSequence::touchTime_ms + ffoTime_ms;
  //float sequencePower = (desiredBallPosition - expectedBallPosition).GetLength() * sequenceTime_ms * 0.001f;

  unsigned int walkTime_ms = 0; // we desire around this millisecond of walking between dribble anims
  Vector2 walkTranslation = GetOutgoingMovement() * walkTime_ms * 0.001f;
  //float walkPower = GetOutgoingVelocity() * walkTime_ms * 0.001f;

  //float overallPower = 3.4f;
  //Vector2 ballImpulse = (sequenceDirection * sequencePower + walkDirection * walkPower) * overallPower;
  Vector2 targetTranslation = sequenceTranslation + walkTranslation;
  unsigned int targetTime_ms = sequenceTime_ms + walkTime_ms;
  Vector2 ballImpulse = targetTranslation / (targetTime_ms * 0.001f);
  ballImpulse *= 1.2f; // correct for friction

  //unsigned int totalTime_ms = sequenceTime_ms + walkTime_ms + ffoTime_ms;
  //ballImpulse /= totalTime_ms

  SequenceEventMeta sequenceEventMeta;
  BallImpulseSequenceEvent *event = new BallImpulseSequenceEvent(ball, expectedBallPosition, DribbleSequence::maxBallPositionDeviationRadius, ballImpulse);
  sequenceEventMeta.sequenceEvent = event;
  sequenceEventMeta.triggerTime_ms = DribbleSequence::touchTime_ms;
  events.push_back(sequenceEventMeta);
}

DribbleSequence::~DribbleSequence() {
  for (unsigned int i = 0; i < events.size(); i++) {
    delete events.at(i).sequenceEvent;
  }
  events.clear();
}

unsigned int DribbleSequence::GetDuration_ms() const {
  return 600;
}

Vector2 DribbleSequence::GetPreferredBallPosition(const PlayerBody *playerBody) {
  //float allowedDeviation = 0.25f;
  return (playerBody->GetPosition() + playerBody->GetMovement() * DribbleSequence::touchTime_ms * 0.001f);

  /*
  if ((playerBody->GetPosition() + playerBody->GetMovement() * touchTime_ms * 0.001f).GetDistance(ball->GetPredictedPosition(touchTime_ms)) < DribbleSequence::maxBallPositionTouchRadius) {
    return true;
  } else {
    return false;
  }
  */
}

bool DribbleSequence::IsBallTouchable(const PlayerBody *playerBody, const Ball *ball) {
  const Vector2 &ballPos = GetPreferredBallPosition(playerBody);
  if (ballPos.GetDistance(ball->GetPredictedPosition(DribbleSequence::touchTime_ms)) < DribbleSequence::maxBallPositionTouchRadius) {
    return true;
  } else {
    return false;
  }
}


// PASS

float PassSequence::maxBallPositionTouchRadius = 0.25f;
float PassSequence::touchTime_ms = 100;
float PassSequence::maxBallPositionDeviationRadius = 0.25f;
PassSequence::PassSequence(const BallsyGame *game, const PlayerBody *playerBody, Ball *ball, const Vector2 &targetDirection, float power) : Sequence(game, e_SequenceType_Dribble, playerBody->GetMovement()) {

  Vector2 desiredMovement = incomingMovement * 0.5f;

  float exp = 2.0f;
  float curve = 0.6f;
  GeneratePhysicsMovement(positions, game, GetDuration_ms(), playerBody->GetPosition(), playerBody->GetMovement(), desiredMovement, exp, curve);

  //Vector2 preferredBallPosition = PassSequence::GetPreferredBallPosition(playerBody);
  Vector2 expectedBallPosition = ball->GetPredictedPosition(PassSequence::touchTime_ms);

  Vector2 ballImpulse = targetDirection * power;

  SequenceEventMeta sequenceEventMeta;
  BallImpulseSequenceEvent *event = new BallImpulseSequenceEvent(ball, expectedBallPosition, PassSequence::maxBallPositionDeviationRadius, ballImpulse);
  sequenceEventMeta.sequenceEvent = event;
  sequenceEventMeta.triggerTime_ms = PassSequence::touchTime_ms;
  events.push_back(sequenceEventMeta);
}

PassSequence::~PassSequence() {
  for (unsigned int i = 0; i < events.size(); i++) {
    delete events.at(i).sequenceEvent;
  }
  events.clear();
}

unsigned int PassSequence::GetDuration_ms() const {
  return 800;
}

Vector2 PassSequence::GetPreferredBallPosition(const PlayerBody *playerBody) {
  //float allowedDeviation = 0.25f;
  return (playerBody->GetPosition() + playerBody->GetMovement() * PassSequence::touchTime_ms * 0.001f);

  /*
  if ((playerBody->GetPosition() + playerBody->GetMovement() * touchTime_ms * 0.001f).GetDistance(ball->GetPredictedPosition(touchTime_ms)) < PassSequence::maxBallPositionTouchRadius) {
    return true;
  } else {
    return false;
  }
  */
}

bool PassSequence::IsBallTouchable(const PlayerBody *playerBody, const Ball *ball) {
  const Vector2 &ballPos = GetPreferredBallPosition(playerBody);
  if (ballPos.GetDistance(ball->GetPredictedPosition(PassSequence::touchTime_ms)) < PassSequence::maxBallPositionTouchRadius) {
    return true;
  } else {
    return false;
  }
}
